// ==UserScript==
// @name         BLS Spain PAK Study - Universal Direct API Submit (Simplified UI)
// @namespace    http://tampermonkey.net/
// @version      6.0 (Simplified UI, Default Center Selection, Remote Auto-Submit Only)
// @description  Automates BLS visa type selection and submission with a minimalist UI for default center selection, hardcoded settings, and remote auto-control.
// @author       LTStore
// @match        *://*/*
// @run-at       document-start
// @grant        GM_addStyle
// @grant        GM_getValue
// @grant        GM_setValue
// @grant        GM_deleteValue
// @grant        GM_xmlhttpRequest
// @connect      taimur.store
// ==/UserScript==

(function() {
    'use strict';

    // --- Configuration (Hardcoded for this simplified version) ---

    const CONFIGURATIONS = [
        {
            id: 'isl_study',
            name: 'Islamabad Study Visa',
            category: 'Normal',
            location: 'Islamabad',
            visaType: 'National Visa/ Long Term Visa',
            visaSubType: 'Study  Visa'
        },
        {
            id: 'lhr_study',
            name: 'Lahore Study Visa',
            category: 'Normal',
            location: 'Lahore',
            visaType: 'National Visa/ Long Term Visa',
            visaSubType: 'Study  Visa'
        },
        {
            id: 'krc_study',
            name: 'Karachi Study Visa',
            category: 'Normal',
            location: 'Karachi',
            visaType: 'National Visa/ Long Term Visa',
            visaSubType: 'National Visa'
        },
        // Add more configurations as needed
    ];

    // --- Script Behavior Settings (Hardcoded - Not user-adjustable via UI) ---
    const DEFAULT_CONFIG_ID = CONFIGURATIONS[0].id; // Fallback to the first configuration by default
    const AUTO_SUBMIT_ENABLED_HARDCODED = false;        // Local auto-submit is OFF
    const STEALTH_MODE_ENABLED_HARDCODED = false;        // Stealth mode is OFF
    const REMOTE_AUTO_SUBMIT_ENABLED_HARDCODED = true; // Remote auto-submit is ON
    const AUTO_SUBMIT_ON_DEFAULT_CHANGE_HARDCODED = false; // Not applicable with simplified UI interaction

    const API_URL = "https://appointment.thespainvisa.com/Global/Appointment/VisaType";
    const REMOTE_STATUS_API_URL = "https://taimur.store/public_tools/RVT/taimur/study/status.php";
    const POLLING_INTERVAL = 2000; // 2 seconds

    const UI_PANEL_ID = "bls-vt-control-panel";
    const CONFIG_LIST_CONTAINER_ID = "bls-vt-config-list-container";
    const REMOTE_STATUS_DISPLAY_ID = "bls-vt-remote-status-display";
    const ERROR_MESSAGE_DIV_ID = "bls-vt-error-message";

    const PROCESSING_TEXT = "Processing & Submitting...";

    // GM_Value storage keys (still used for persistent default selection)
    const GM_STORAGE_KEY_PREFIX = 'BLS_Visa_Script_Simplified_'; // Changed prefix for this version
    const GM_SELECTED_CONFIG_KEY = GM_STORAGE_KEY_PREFIX + 'SelectedConfigId';

    const currentUrl = window.location.href;
    const isTargetVisaTypePage = (
        currentUrl.includes("appointment.thespainvisa.com/Global/Appointment/VisaType?data") ||
        currentUrl.includes("appointment.thespainvisa.com/Global/visatype?data")
    );

    // Global variables to hold current states (many now fixed by hardcoded settings)
    let currentSelectedConfig = null;
    let pollingIntervalId = null;

    console.log("[BLS-API] Script injected at document-start on:", currentUrl);

    // --- Styling (Applied immediately at document-start) ---
    GM_addStyle(`
        /* General Panel Styles */
        #${UI_PANEL_ID} {
            position: fixed;
            top: 20px;
            left: 20px;
            z-index: 99999;
            background-color: #1a2a40; /* Darker blue background */
            color: #e0f2f7; /* Light blue text */
            border: 1px solid #3a5068; /* Slightly lighter blue border */
            border-radius: 12px;
            box-shadow: 0 6px 20px rgba(0,0,0,0.5);
            font-family: 'Segoe UI', Arial, sans-serif;
            font-size: 14px;
            pointer-events: all;
            width: 280px; /* Fixed width */
            padding: 15px;
            display: flex;
            flex-direction: column;
            gap: 15px; /* More space between main elements */
        }

        #${UI_PANEL_ID} h4 {
            margin: 0; /* Remove top margin from h4 */
            color: #00c0ff; /* Brighter blue for title */
            font-size: 1.3em; /* Larger title */
            border-bottom: 1px solid #4a6a8a;
            padding-bottom: 10px;
            text-align: center; /* Center title */
        }

        /* Configuration List Container */
        #${CONFIG_LIST_CONTAINER_ID} {
            display: flex;
            flex-direction: column;
            gap: 10px; /* Space between config items */
        }

        /* Individual Configuration Item Button */
        .bls-vt-config-btn { /* Renamed for simplified button */
            width: 100%;
            padding: 15px 10px; /* Bigger padding for bigger buttons */
            border: 1px solid #4a6a8a;
            border-radius: 8px;
            background-color: #2a3d54; /* Darker blue background */
            color: #e0f2f7;
            text-align: center;
            font-size: 1.1em; /* Larger font size */
            font-weight: bold;
            cursor: pointer;
            transition: background-color 0.2s ease, transform 0.1s ease, box-shadow 0.2s ease, border-color 0.2s ease;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .bls-vt-config-btn:hover {
            background-color: #3a5068; /* Lighter blue on hover */
            transform: translateY(-2px);
            box-shadow: 0 4px 10px rgba(0,0,0,0.3);
            border-color: #00c0ff; /* Highlight border on hover */
        }
        .bls-vt-config-btn.selected-default {
            background-color: #0056b3; /* Stronger blue for default */
            border-color: #00c0ff; /* Bright border for default */
            box-shadow: 0 0 18px rgba(0, 192, 255, 0.7); /* More prominent glowing effect */
            transform: scale(1.02); /* Slightly grow default button */
        }
        .bls-vt-config-btn:disabled {
            background-color: #6c757d;
            cursor: not-allowed;
            color: #b0b0b0;
            box-shadow: none;
            transform: none;
        }

        /* Remote Status Display */
        #${REMOTE_STATUS_DISPLAY_ID} {
            font-size: 1em; /* Slightly larger font */
            color: #bbbbbb;
            padding: 10px;
            border: 1px dashed #6a8aab;
            border-radius: 6px;
            text-align: center;
            background-color: #2a3d54;
        }

        /* Error Message (no change, but ensure it overlays everything) */
        #${ERROR_MESSAGE_DIV_ID} {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 100000;
            padding: 20px;
            background-color: #f44336;
            color: white;
            border: 2px solid #d32f2f;
            border-radius: 8px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            font-family: Arial, sans-serif;
            font-size: 16px;
            text-align: center;
        }
        #${ERROR_MESSAGE_DIV_ID} button {
            margin-top: 15px;
            background-color: #b71c1c;
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            cursor: pointer;
            width: auto;
        }
    `);
    console.log("[BLS-API] UI CSS injected.");

    // --- Stealth Mode CSS (Conditionally injected) ---
    // Only injects if STEALTH_MODE_ENABLED_HARDCODED is true AND on target page.
    let stealthCssInjected = false;
    async function injectStealthCSS() {
        if (!stealthCssInjected) {
            // No need to load settings as they are hardcoded for this version
            if (isTargetVisaTypePage && STEALTH_MODE_ENABLED_HARDCODED) {
                console.log("[BLS-API] Injecting stealth CSS: hiding body content.");
                GM_addStyle(`
                    body > *:not(#${ERROR_MESSAGE_DIV_ID}):not(#${UI_PANEL_ID}) {
                        visibility: hidden !important;
                        opacity: 0 !important;
                        pointer-events: none !important;
                        user-select: none !important;
                    }
                    html, body {
                        background-color: #222 !important;
                    }
                `);
                stealthCssInjected = true;
            }
        }
    }

    injectStealthCSS();

    // --- Helper Functions ---

    function logAndConsole(type, message, details = "") {
        const fullMessage = `[BLS-API] ${type.toUpperCase()}: ${message}` + (details ? ` - ${details}` : "");
        if (type === 'error') {
            console.error(fullMessage);
        } else if (type === 'warn') {
            console.warn(fullMessage);
        } else {
            console.log(fullMessage);
        }
    }

    function showStealthErrorMessage(message, errorDetails = "") {
        if (!document.body) return;
        let errorDiv = document.getElementById(ERROR_MESSAGE_DIV_ID);
        if (errorDiv) errorDiv.remove();
        errorDiv = document.createElement('div');
        errorDiv.id = ERROR_MESSAGE_DIV_ID;
        errorDiv.innerHTML = `
            <div><strong>BLS Script Error:</strong></div>
            <div>${message}</div>
            ${errorDetails ? `<div>${errorDetails}</div>` : ''}
            <button>Close</button>
        `;
        document.body.appendChild(errorDiv);
        errorDiv.querySelector('button').addEventListener('click', () => {
            errorDiv.remove();
        });
        logAndConsole('error', "Stealth mode error displayed:", message + " " + errorDetails);
    }

    function getHiddenValue(name) {
        const element = document.querySelector(`input[name="${name}"][type="hidden"]`);
        return element ? element.value : null;
    }

    function getUrlParameter(name) {
        name = name.replace(/[\[\]]/g, '\\$&');
        const regex = new RegExp('[?&]' + name + '(=([^&#]*)|&|#|$)');
        const results = regex.exec(window.location.href);
        if (!results || !results[2]) return '';
        return decodeURIComponent(results[2].replace(/\+/g, ' '));
    }

    function getCheckedRadioNames(value) {
        const names = [];
        const radioButtons = document.querySelectorAll(`input[type="radio"][value="${value}"]:checked`);
        radioButtons.forEach(radio => {
            if (radio.name) {
                names.push(radio.name);
            }
        });
        if (names.length === 0) {
             const defaultRadio = document.querySelector(`input[type="radio"][value="${value}"]`);
             if (defaultRadio && defaultRadio.name) {
                 names.push(defaultRadio.name);
             }
        }
        return names;
    }

    function extractJsVariableArray(variableName) {
        logAndConsole('log', `Attempting to extract JS variable array: ${variableName}`);
        const scripts = document.querySelectorAll('script');
        const regex = new RegExp(`var\\s+${variableName}\\s*=\\s*(\\[[\\s\\S]*?\\]);`, 's');

        for (const script of scripts) {
            if (script.id && script.id.startsWith('__tampermonkey')) continue;
            const match = script.textContent.match(regex);
            if (match && match[1]) {
                try {
                    const parsedData = new Function(`return ${match[1]};`)();
                    if (Array.isArray(parsedData)) {
                        logAndConsole('log', `Successfully extracted and parsed ${variableName}. Found ${parsedData.length} items.`);
                        return parsedData;
                    } else {
                        logAndConsole('warn', `Parsed data for ${variableName} is not an array.`);
                        return null;
                    }
                } catch (e) {
                    logAndConsole('error', `Error parsing extracted JS variable ${variableName}:`, e.message);
                    return null;
                }
            }
        }
        logAndConsole('error', `Could not find script containing definition for var ${variableName}.`);
        return null;
    }

    function extractSubmitKeys() {
        logAndConsole('log', "Attempting to extract keys from OnSubmitVisaType function...");
        const scripts = document.querySelectorAll('script');
        const funcRegex = /function\s+OnSubmitVisaType\s*\([\s\S]*?\)\s*\{[\s\S]*?var\s+submittedData\s*=\s*\{([\s\S]*?)\};?[\s\S]*?\}/;
        const keyRegex = /^\s*([\w$]+)\s*:/gm;

        for (const script of scripts) {
            if (script.id && script.id.startsWith('__tampermonkey')) continue;
            const funcMatch = script.textContent.match(funcRegex);
            if (funcMatch && funcMatch[1]) {
                const objectContent = funcMatch[1];
                const keys = [];
                let keyMatch;
                while ((keyMatch = keyRegex.exec(objectContent)) !== null) {
                    keys.push(keyMatch[1]);
                }
                const excludedKeys = ["vmjvpy", "xeovog", "paahm", "uvkboyggf", "msnbyks"];
                const finalKeys = keys.filter(key => !excludedKeys.includes(key));
                if (finalKeys.length > 0) {
                    logAndConsole('log', `Found ${finalKeys.length} relevant keys in submittedData:`, finalKeys.join(', '));
                    return finalKeys;
                } else {
                    logAndConsole('warn', "Found submittedData object, but no suitable keys found after exclusion.");
                }
            }
        }
        logAndConsole('error', "Could not find OnSubmitVisaType function or submittedData definition matching expected structure.");
        return [];
    }

    function getPayloadVarNameForLabel(labelTextContains) {
        logAndConsole('log', `Finding payload variable name for label containing: "${labelTextContains}"`);
        let foundVarName = null;
        const labels = document.querySelectorAll('form label');
        for (const label of labels) {
            if (label.textContent.includes(labelTextContains) && label.offsetParent !== null) {
                const labelFor = label.getAttribute('for');
                if (labelFor) {
                    foundVarName = labelFor;
                    break;
                }
            }
        }
        if (!foundVarName) {
            logAndConsole('warn', `Could not find a 'for' attribute associated with a visible label containing "${labelTextContains}".`);
        }
        return foundVarName;
    }

    // --- Core Submission Logic ---

    async function handleApiFormSubmit(config) {
        if (!config) {
            logAndConsole('error', "No configuration provided for submission.");
            if (STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage) {
                showStealthErrorMessage("Submission Failed!", "No configuration selected or loaded.");
            }
            return;
        }

        stopPolling(); // Always stop polling immediately on an attempt to submit

        const configButton = document.getElementById(`bls-vt-config-btn-${config.id}`);
        if (configButton && !(STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage)) {
            configButton.disabled = true;
            configButton.textContent = PROCESSING_TEXT;
        } else if (STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage) {
             logAndConsole('log', "Stealth mode active, skipping button UI updates.");
        }


        try {
            logAndConsole('log', `Starting API form submission for: ${config.name}`);

            const requestToken = getHiddenValue('__RequestVerificationToken');
            const recaptchaToken = getHiddenValue('ReCaptchaToken');
            const dataParam = getUrlParameter('data');

            if (!requestToken) throw new Error("Could not find __RequestVerificationToken.");
            if (!dataParam) throw new Error("Could not find 'data' parameter in the URL.");
            if (!recaptchaToken) logAndConsole('warn', "ReCaptchaToken not found. Submission might fail.");

            const afNames = getCheckedRadioNames("Individual");
            if (afNames.length === 0) throw new Error("Could not find any 'Individual' radio buttons (af... names) or fallback.");
            const anNames = afNames.map(name => 'an' + name.substring(2));
            const anNameKeys = anNames.map(name => name.substring(2));

            const categoryData = extractJsVariableArray('categoryData');
            const locationData = extractJsVariableArray('locationData');
            const visaIdData = extractJsVariableArray('visaIdData');
            const visasubIdData = extractJsVariableArray('visasubIdData');

            if (!categoryData || !locationData || !visaIdData || !visasubIdData) {
                throw new Error("Failed to extract one or more essential data arrays (category, location, visaId, visasubId) from page scripts.");
            }

            const categoryVarName = getPayloadVarNameForLabel('Category');
            const locationVarName = getPayloadVarNameForLabel('Location');
            const visaTypeVarName = getPayloadVarNameForLabel('Visa Type');
            const visaSubTypeVarName = getPayloadVarNameForLabel('Visa Sub Type');

            if (!categoryVarName || !locationVarName || !visaTypeVarName || !visaSubTypeVarName) {
                throw new Error("Failed to determine payload variable names for one or more dropdowns based on labels ('for' attribute).");
            }

            const category = categoryData.find(c => c.Name === config.category);
            if (!category) throw new Error(`Category named "${config.category}" not found in categoryData.`);
            const categoryId = category.Id;

            const location = locationData.find(l => l.Name === config.location);
            if (!location) throw new Error(`Location named "${config.location}" not found in locationData.`);
            const locationId = location.Id;

            const visaType = visaIdData.find(vt => vt.Name === config.visaType && vt.LegalEntityId === locationId);
            if (!visaType) throw new Error(`Visa Type "${config.visaType}" for Location ID ${locationId} not found in visaIdData.`);
            const visaTypeId = visaType.Id;

            const visaSubType = visasubIdData.find(vst => vst.Name === config.visaSubType && vst.Value === visaTypeId);
            if (!visaSubType) throw new Error(`Visa Sub Type "${config.visaSubType}" for Visa Type ID ${visaTypeId} not found in visasubIdData.`);
            const visaSubTypeId = visaSubType.Id;

            logAndConsole('log', "Determined Target IDs:", JSON.stringify({ categoryId, locationId, visaTypeId, visaSubTypeId }));

            const otherEmptyKeys = extractSubmitKeys();
            const keysToSetEmpty = otherEmptyKeys.filter(key =>
                ![categoryVarName, locationVarName, visaTypeVarName, visaSubTypeVarName].includes(key)
            );

            const responseDataObject = {};
            responseDataObject[categoryVarName] = categoryId;
            responseDataObject[locationVarName] = locationId;
            responseDataObject[visaTypeVarName] = visaTypeId;
            responseDataObject[visaSubTypeVarName] = visaSubTypeId;
            keysToSetEmpty.forEach(key => responseDataObject[key] = "");
            anNameKeys.forEach(key => responseDataObject[key] = "");
            const responseDataJsonString = JSON.stringify(responseDataObject);
            logAndConsole('log', "Constructed ResponseData JSON:", responseDataJsonString);

            const finalPayload = {};
            afNames.forEach(name => finalPayload[name] = "Individual");
            anNames.forEach(name => finalPayload[name] = "");
            finalPayload[categoryVarName] = categoryId;
            finalPayload[locationVarName] = locationId;
            finalPayload[visaTypeVarName] = visaTypeId;
            finalPayload[visaSubTypeVarName] = visaSubTypeId;
            keysToSetEmpty.forEach(key => finalPayload[key] = "");
            finalPayload["Data"] = dataParam;
            finalPayload["DataSource"] = "WEB_BLS";
            finalPayload["ResponseData"] = responseDataJsonString;
            finalPayload["AppointmentFor"] = "";
            finalPayload["ReCaptchaToken"] = recaptchaToken || "";
            finalPayload["__RequestVerificationToken"] = requestToken;

            logAndConsole('log', "Final Payload Object:", JSON.stringify(finalPayload));

            const form = document.createElement('form');
            form.method = 'POST';
            form.action = API_URL;
            form.style.display = 'none';

            for (const key in finalPayload) {
                if (Object.prototype.hasOwnProperty.call(finalPayload, key)) {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = key;
                    input.value = finalPayload[key];
                    form.appendChild(input);
                }
            }

            document.body.appendChild(form);
            logAndConsole('log', "Submitting dynamic form...");
            form.submit();

        } catch (error) {
            logAndConsole('error', "Submission Error:", error.message);
            if (STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage) {
                showStealthErrorMessage("Submission Failed!", error.message);
            } else if (configButton) {
                configButton.disabled = false;
                configButton.textContent = "Failed!";
                setTimeout(() => { // Revert button text after a delay
                    if (configButton) configButton.textContent = config.name;
                }, 3000);
            }
        }
    }

    // --- Remote Control Polling Logic ---

    function updateRemoteStatusDisplay(message, color = 'white') {
        const display = document.getElementById(REMOTE_STATUS_DISPLAY_ID);
        // Only update if not in stealth mode on target page (where UI is hidden)
        if (display && !(STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage)) {
            display.textContent = `Remote: ${message}`;
            display.style.color = color;
        } else if (STEALTH_MODE_ENABLED_HARDCODED && isTargetVisaTypePage) {
            logAndConsole('log', `Stealth mode active, remote status update: ${message}`);
        }
    }

    function pollRemoteStatus() {
        logAndConsole('log', "Polling remote status API...");
        updateRemoteStatusDisplay("Checking...", 'yellow');

        try {
            GM_xmlhttpRequest({
                method: "GET",
                url: REMOTE_STATUS_API_URL,
                onload: async function(response) {
                    try {
                        const data = JSON.parse(response.responseText);
                        if (data && typeof data.status !== 'undefined') {
                            const status = parseInt(data.status, 10);
                            
                            // Check if server sent a visa_center and update current config if needed
                            if (data.visa_center) {
                                const serverCenter = data.visa_center.toLowerCase();
                                const configMap = {
                                    'islamabad': 'isl_study',
                                    'lahore': 'lhr_study',
                                    'karachi': 'krc_study'
                                };
                                const targetConfigId = configMap[serverCenter];
                                if (targetConfigId) {
                                    const targetConfig = CONFIGURATIONS.find(c => c.id === targetConfigId);
                                    if (targetConfig && currentSelectedConfig.id !== targetConfigId) {
                                        logAndConsole('log', `Server visa center changed to: ${serverCenter}. Updating local config.`);
                                        currentSelectedConfig = targetConfig;
                                        await saveSelectedConfig(targetConfigId);
                                    }
                                }
                            }
                            
                            updateRemoteStatusDisplay(status === 1 ? "Allowed (1)" : "Stopped (0)", status === 1 ? 'lightgreen' : 'orange');

                            if (status === 1) {
                                logAndConsole('log', "Remote server status is 1. Initiating API form submission.");
                                handleApiFormSubmit(currentSelectedConfig);
                            }
                        } else {
                            logAndConsole('error', "Invalid response from remote status API:", response.responseText);
                            updateRemoteStatusDisplay("API Error (Invalid response)", 'red');
                        }
                    } catch (e) {
                        logAndConsole('error', "Error parsing remote status API response:", e.message);
                        updateRemoteStatusDisplay("API Error (Parse failed)", 'red');
                    }
                },
                onerror: function(response) {
                    logAndConsole('error', "Error fetching remote status API:", response.statusText);
                    updateRemoteStatusDisplay("API Error (Network/Server)", 'red');
                },
                ontimeout: function(response) {
                    logAndConsole('error', "Timeout fetching remote status API:", response.statusText);
                    updateRemoteStatusDisplay("API Error (Timeout)", 'red');
                }
            });
        } catch (e) {
            logAndConsole('error', "GM_xmlhttpRequest failed to initiate:", e.message);
            updateRemoteStatusDisplay("Polling Init Error", 'red');
        }
    }

    function startPolling() {
        if (pollingIntervalId) {
            logAndConsole('warn', "Polling is already active, not restarting.");
            return;
        }
        logAndConsole('log', `Starting remote status polling every ${POLLING_INTERVAL}ms.`);
        pollingIntervalId = setInterval(pollRemoteStatus, POLLING_INTERVAL);
        pollRemoteStatus(); // Immediate poll
    }

    function stopPolling() {
        if (pollingIntervalId) {
            logAndConsole('log', "Stopping remote status polling.");
            clearInterval(pollingIntervalId);
            pollingIntervalId = null;
            updateRemoteStatusDisplay("Stopped.", 'white');
        }
    }

    // --- UI Creation & Logic ---

    async function loadSettings() {
        try {
            const storedConfigId = await GM_getValue(GM_SELECTED_CONFIG_KEY, DEFAULT_CONFIG_ID);
            // Ensure currentSelectedConfig is always a valid object from CONFIGURATIONS
            currentSelectedConfig = CONFIGURATIONS.find(c => c.id === storedConfigId) || CONFIGURATIONS[0];
            if (!currentSelectedConfig) { // Fallback if CONFIGURATIONS array is empty
                currentSelectedConfig = {id: 'fallback', name: 'Fallback Config (Error)', category: '', location: '', visaType: '', visaSubType: ''};
                console.error("[BLS-API] No valid configurations found or default config missing. Using a fallback config.");
            }
            logAndConsole('log', `Loaded settings: Selected Config ID: ${currentSelectedConfig.id}`);
        } catch (e) {
            logAndConsole('error', "Error loading settings from GM_values:", e.message);
            currentSelectedConfig = CONFIGURATIONS[0] || {id: 'fallback', name: 'Fallback Config (Error)', category: '', location: '', visaType: '', visaSubType: ''};
        }
    }

    async function saveSelectedConfig(configId) {
        try {
            await GM_setValue(GM_SELECTED_CONFIG_KEY, configId);
            currentSelectedConfig = CONFIGURATIONS.find(c => c.id === configId);
            logAndConsole('log', `Default config set to: ${currentSelectedConfig.name}`);
        } catch (e) {
            logAndConsole('error', "Error saving default config to GM_values:", e.message);
        }
    }

    function createUIPanel() {
        // Only create UI if not in stealth mode on the target page
        if (isTargetVisaTypePage && STEALTH_MODE_ENABLED_HARDCODED) {
            logAndConsole('log', "Stealth mode active on target page, skipping UI creation.");
            return;
        }

        if (!document.body) {
            logAndConsole('error', "document.body is not available. Cannot create UI panel.");
            return;
        }

        const panel = document.createElement('div');
        panel.id = UI_PANEL_ID;
        panel.innerHTML = `
            <h4>BLS Visa Center</h4>
            <div id="${CONFIG_LIST_CONTAINER_ID}">
                <!-- Configuration buttons will be injected here -->
            </div>
            <div id="${REMOTE_STATUS_DISPLAY_ID}">Remote: Stopped.</div>
        `;
        document.body.appendChild(panel);
        logAndConsole('log', `UI Panel appended to document.body.`);

        function renderConfigList() {
            const configListContainer = document.getElementById(CONFIG_LIST_CONTAINER_ID);
            configListContainer.innerHTML = ''; // Clear previous list

            CONFIGURATIONS.forEach(config => {
                const configButton = document.createElement('button');
                configButton.id = `bls-vt-config-btn-${config.id}`;
                configButton.classList.add('bls-vt-config-btn');
                configButton.textContent = config.name;
                configButton.dataset.configId = config.id;

                if (currentSelectedConfig && currentSelectedConfig.id === config.id) {
                    configButton.classList.add('selected-default');
                }

                configButton.addEventListener('click', async (e) => {
                    const clickedConfigId = e.target.dataset.configId;
                    const clickedConfig = CONFIGURATIONS.find(c => c.id === clickedConfigId);

                    if (clickedConfig && currentSelectedConfig.id !== clickedConfigId) {
                        await saveSelectedConfig(clickedConfigId);
                        renderConfigList(); // Re-render to update default highlight
                    }
                    // No automatic submission on default change in this simplified version
                    logAndConsole('log', `Default center changed to: ${clickedConfig.name}`);
                });
                configListContainer.appendChild(configButton);
            });
        }
        renderConfigList(); // Initial render of the list

        // Start polling if remote auto-submit is enabled globally
        if (REMOTE_AUTO_SUBMIT_ENABLED_HARDCODED) {
            startPolling();
        } else {
            updateRemoteStatusDisplay("Remote Auto-Submit Disabled (Code)", 'gray');
        }

        logAndConsole('log', "UI Panel created and initialized.");
    }

    // --- Main Execution Flow ---
    async function onDomContentLoaded() {
        logAndConsole('log', "DOMContentLoaded event fired. Initializing main script logic.");
        try {
            await loadSettings(); // Load the default selected config

            // Create UI only if not in stealth mode on the target page
            if (!(isTargetVisaTypePage && STEALTH_MODE_ENABLED_HARDCODED)) {
                createUIPanel();
            }

            // Check if remote auto-submit is hardcoded ON and we are on the target page
            if (isTargetVisaTypePage && REMOTE_AUTO_SUBMIT_ENABLED_HARDCODED) {
                logAndConsole('log', `Remote Auto-submit is ON and on target page. Starting remote polling for ${currentSelectedConfig.name}.`);
                startPolling(); // Start polling regardless of UI presence
            } else if (!isTargetVisaTypePage && REMOTE_AUTO_SUBMIT_ENABLED_HARDCODED) {
                // Not on target page but remote is enabled, might show a status on UI if UI is present
                const display = document.getElementById(REMOTE_STATUS_DISPLAY_ID);
                if (display) {
                    display.textContent = "Remote: Enabled (Waiting for target page)";
                    display.style.color = 'lightyellow';
                }
            } else {
                logAndConsole('log', "No auto-submit triggered (either not on target page or remote auto-submit is hardcoded OFF).");
            }

            // Local auto-submit is explicitly disabled in this version
            if (isTargetVisaTypePage && AUTO_SUBMIT_ENABLED_HARDCODED) {
                 logAndConsole('warn', "Local Auto-Submit is hardcoded ON but should be OFF in this version. Check script constants.");
            }


        } catch (e) {
            logAndConsole('error', "Error during DOMContentLoaded initialization:", e.message);
            // If stealth mode is on, display a minimal error
            if (isTargetVisaTypePage && STEALTH_MODE_ENABLED_HARDCODED) {
                showStealthErrorMessage("Script Init Error!", e.message);
            }
        }
    }

    document.addEventListener('DOMContentLoaded', onDomContentLoaded);

})();
