<?php
// RVT/panel.php
include 'config.php';

$client = isset($_GET['client']) ? $_GET['client'] : '';
$category = isset($_GET['category']) ? $_GET['category'] : '';

if (!$client || !$category) {
    die("Invalid Link. Please specify client and category.");
}

$settings = getPanelSettings($client, $category);

if (!$settings) {
    die("Panel not found. Please contact admin.");
}

$status = (int)$settings['status'];
$visa_center = $settings['visa_center'];

// Theme Logic based on category/client (mimicking your old styles)
$bgColor = '#0f0f1d'; // Default dark
$accentColor = '#9333ea'; // Default purple
$btnColor = '#32cd32';

if ($client == 'bilal' && $category == 'work') {
    $bgColor = '#331a00'; $accentColor = '#ff8c00';
} elseif ($client == 'taimur' && $category == 'study') {
    $bgColor = '#1b4200'; $accentColor = '#00bfff';
} elseif ($client == 'taimur' && $category == 'work') {
    $bgColor = '#001a33'; $accentColor = '#0095ff';
} elseif ($client == 'bilal' && $category == 'study') {
    $bgColor = '#1a0033'; $accentColor = '#9333ea';
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RVT Panel - <?php echo htmlspecialchars(ucfirst($client) . ' ' . ucfirst($category)); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: <?php echo $bgColor; ?>;
            color: #e0e0e0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            margin: 0;
            overflow: hidden;
            position: relative;
        }
        /* Glowing grid effect */
        body::before {
            content: '';
            position: absolute;
            top: 0; left: 0; width: 100%; height: 100%;
            background-image: linear-gradient(to right, rgba(255,255,255,0.05) 1px, transparent 1px),
                              linear-gradient(to bottom, rgba(255,255,255,0.05) 1px, transparent 1px);
            background-size: 20px 20px;
            animation: grid-glow 5s infinite alternate;
            opacity: 0.5; z-index: -1;
        }
        @keyframes grid-glow { from { filter: brightness(0.8); } to { filter: brightness(1.2); } }

        .container {
            background-color: rgba(0,0,0,0.6);
            border: 1px solid <?php echo $accentColor; ?>;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 0 25px <?php echo $accentColor; ?>66;
            text-align: center;
            width: 90%; max-width: 400px;
            position: relative; z-index: 1;
            backdrop-filter: blur(10px);
        }
        h1 {
            color: <?php echo $accentColor; ?>;
            margin-bottom: 25px;
            text-shadow: 0 0 10px <?php echo $accentColor; ?>99;
            font-size: 2em;
            text-transform: capitalize;
        }
        .status-display {
            margin-bottom: 30px; font-size: 1.8em; font-weight: bold;
            padding: 15px; border-radius: 8px;
            background-color: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
        }
        .status-text { color: #ff4500; }
        .status-text.active { color: #32cd32; }
        
        .buttons-group { display: flex; flex-direction: column; gap: 15px; }
        button {
            padding: 15px 25px; border: none; border-radius: 8px;
            font-size: 1.1em; cursor: pointer; transition: all 0.3s ease;
            text-transform: uppercase; letter-spacing: 1px; font-weight: bold;
            box-shadow: 0 4px 15px rgba(0,0,0,0.3);
        }
        button:hover { filter: brightness(1.1); transform: translateY(-2px); }
        
        #allowBtn { background-color: #32cd32; color: #0f0f1d; }
        #stopBtn { background-color: #ff4500; color: #0f0f1d; }
        #allowOnceBtn { background-color: <?php echo $accentColor; ?>; color: #fff; }
        #delayedAllowBtn { background-color: #e0e0e0; color: #333; }
        #changeCenterBtn { background-color: #32cd32; color: #0f0f1d; }

        .loading-overlay {
            position: absolute; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: none; justify-content: center; align-items: center;
            z-index: 10; border-radius: 12px; font-size: 1.5em; color: <?php echo $accentColor; ?>;
        }
        .loading-overlay.visible { display: flex; }
        .loader {
            border: 4px solid #f3f3f3; border-top: 4px solid <?php echo $accentColor; ?>;
            border-radius: 50%; width: 30px; height: 30px;
            animation: spin 1s linear infinite; margin-right: 10px;
        }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    </style>
</head>
<body>
    <div class="container">
        <h1><?php echo htmlspecialchars($client . ' ' . $category); ?></h1>
        <div class="status-display">Current Status: <span id="currentStatus" class="status-text <?php echo $status == 1 ? 'active' : ''; ?>"><?php echo $status == 1 ? 'ALLOWED' : 'STOPPED'; ?></span></div>
        
        <div class="buttons-group">
            <button id="allowBtn">Allow</button>
            <button id="stopBtn">Stop</button>
            <button id="allowOnceBtn">Allow Once (5s)</button>
            <button id="delayedAllowBtn">Delayed Allow</button>
            <button id="changeCenterBtn">Change Center</button>
        </div>

        <div style="margin-top:16px;">
            <div id="countdownContainer" style="display:none; align-items:center; gap:8px; justify-content:center;">
                <div id="countdownBar" style="width:200px; height:12px; background:#222; border-radius:6px; overflow:hidden;">
                    <div id="countdownFill" style="height:100%; width:0%; background:#32cd32;"></div>
                </div>
                <div id="countdownText" style="font-size:0.9em; color:#cfd8dc;">0s</div>
            </div>
        </div>

        <div style="margin-top:12px; padding:8px; background:rgba(0,0,0,0.3); border-radius:6px; font-size:0.9em;">
            Current Center: <span id="currentCenter" style="color:<?php echo $accentColor; ?>; font-weight:bold; text-transform:capitalize;"><?php echo htmlspecialchars($visa_center); ?></span>
        </div>
        
        <div style="position: absolute; top: 12px; right: 12px;">
            <button id="configureBtn" style="padding:6px 10px; font-size:0.9em;">Config</button>
        </div>

        <div id="loadingOverlay" class="loading-overlay">
            <div class="loader"></div> Processing...
        </div>
    </div>

    <script>
        // Pass PHP variables to JS
        const CLIENT = "<?php echo $client; ?>";
        const CATEGORY = "<?php echo $category; ?>";
        const API_URL = "api.php";

        const currentStatusSpan = document.getElementById('currentStatus');
        const loadingOverlay = document.getElementById('loadingOverlay');
        
        // Helper: Show Toast
        function showToast(message, type = 'info') {
            const t = document.createElement('div');
            t.textContent = message;
            t.style.cssText = `position:fixed; right:12px; bottom:12px; background:${type==='error'?'#ff4d4f':'#333'}; color:#fff; padding:10px 14px; border-radius:6px; z-index:9999; box-shadow:0 4px 10px rgba(0,0,0,0.3);`;
            document.body.appendChild(t);
            setTimeout(() => t.remove(), 3000);
        }

        function showLoading() { loadingOverlay.classList.add('visible'); }
        function hideLoading() { loadingOverlay.classList.remove('visible'); }

        async function sendAction(action, extraData = {}) {
            showLoading();
            try {
                const formData = new FormData();
                formData.append('action', action);
                formData.append('client', CLIENT);
                formData.append('category', CATEGORY);
                for (const key in extraData) {
                    formData.append(key, extraData[key]);
                }

                const res = await fetch(API_URL, { method: 'POST', body: formData });
                const data = await res.json();

                if (data.success) {
                    if (data.newStatus !== undefined) {
                        currentStatusSpan.textContent = data.newStatus == 1 ? 'ALLOWED' : 'STOPPED';
                        currentStatusSpan.className = 'status-text ' + (data.newStatus == 1 ? 'active' : '');
                    }
                    if (data.visa_center) {
                        document.getElementById('currentCenter').textContent = data.visa_center;
                    }
                    showToast(data.message || 'Success');
                    
                    // Handle Allow Once Countdown
                    if (action === 'allow_once' && data.allow_once_duration_ms) {
                        startCountdown(data.allow_once_duration_ms);
                        setTimeout(() => sendAction('stop'), data.allow_once_duration_ms);
                    }
                } else {
                    showToast(data.message || 'Error', 'error');
                }
            } catch (e) {
                showToast('Network Error', 'error');
            } finally {
                hideLoading();
            }
        }

        // Countdown Logic
        let countdownInterval;
        function startCountdown(ms) {
            const container = document.getElementById('countdownContainer');
            const fill = document.getElementById('countdownFill');
            const text = document.getElementById('countdownText');
            container.style.display = 'flex';
            
            const start = Date.now();
            if (countdownInterval) clearInterval(countdownInterval);
            
            countdownInterval = setInterval(() => {
                const elapsed = Date.now() - start;
                const pct = Math.min(100, (elapsed / ms) * 100);
                fill.style.width = pct + '%';
                text.textContent = Math.ceil((ms - elapsed)/1000) + 's';
                
                if (elapsed >= ms) {
                    clearInterval(countdownInterval);
                    container.style.display = 'none';
                }
            }, 100);
        }

        // Event Listeners
        document.getElementById('allowBtn').addEventListener('click', () => sendAction('allow'));
        document.getElementById('stopBtn').addEventListener('click', () => sendAction('stop'));
        document.getElementById('delayedAllowBtn').addEventListener('click', () => sendAction('delayed_allow'));
        
        document.getElementById('allowOnceBtn').addEventListener('click', () => {
            // Default 5000ms, server will override
            sendAction('allow_once'); 
        });

        document.getElementById('changeCenterBtn').addEventListener('click', () => {
            const centers = ['islamabad', 'karachi', 'lahore'];
            const current = document.getElementById('currentCenter').textContent.toLowerCase();
            const next = centers[(centers.indexOf(current) + 1) % centers.length];
            sendAction('set_center', { center: next });
        });

        document.getElementById('configureBtn').addEventListener('click', () => {
            const ms = prompt("Enter Delayed MS (e.g. 1300):", "1300");
            if(ms) sendAction('configure', { delayed_ms: ms });
        });

    </script>
</body>
</html>
