<?php
// bls.txyber.com/publictoolsbls_blocked/vt-cp/index.php
include 'config.php';

$conn = getDbConnection();
$status = 0; // Default status
$stmt = $conn->prepare("SELECT status FROM " . DB_TABLE . " WHERE id = 1");
if ($stmt) {
    $stmt->execute();
    $stmt->bind_result($status);
    $stmt->fetch();
    $stmt->close();
}
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BLS Remote Control Panel - Bilal Work</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #331a00; /* Dark orange background */
            color: #e0e0e0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            margin: 0;
            overflow: hidden; /* Prevent scrollbar for futuristic feel */
            position: relative;
        }
        /* Glowing grid effect */
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-image: linear-gradient(to right, rgba(255,140,0,0.05) 1px, transparent 1px),
                              linear-gradient(to bottom, rgba(255,140,0,0.05) 1px, transparent 1px);
            background-size: 20px 20px;
            animation: grid-glow 5s infinite alternate;
            opacity: 0.5;
            z-index: -1;
        }
        @keyframes grid-glow {
            from { filter: brightness(0.8); }
            to { filter: brightness(1.2); }
        }

        .container {
            background-color: #1d0f0a;
            border: 1px solid #764a21;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 0 25px rgba(255, 140, 0, 0.4); /* Orange glowing shadow */
            text-align: center;
            width: 90%;
            max-width: 400px;
            position: relative;
            z-index: 1;
        }
        h1 {
            color: #ff8c00; /* Orange */
            margin-bottom: 25px;
            text-shadow: 0 0 10px rgba(255, 140, 0, 0.6);
            font-size: 2em;
        }
        .status-display {
            margin-bottom: 30px;
            font-size: 1.8em;
            font-weight: bold;
            padding: 15px;
            border-radius: 8px;
            background-color: #321a12;
            border: 1px solid #8a5a3a;
            box-shadow: inset 0 0 10px rgba(0,0,0,0.5);
        }
        .status-text {
            color: #ff4500; /* OrangeRed for Stopped */
        }
        .status-text.active {
            color: #32cd32; /* LimeGreen for Allowed */
        }
        .buttons-group {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        button {
            padding: 15px 25px;
            border: none;
            border-radius: 8px;
            font-size: 1.1em;
            cursor: pointer;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 1px;
            font-weight: bold;
            box-shadow: 0 4px 15px rgba(0,0,0,0.3);
        }
        button:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(255, 140, 0, 0.6);
        }
        #allowBtn {
            background-color: #32cd32; /* LimeGreen */
            color: #0f0f1d;
        }
        #allowBtn:hover {
            background-color: #22a822;
            box-shadow: 0 0 20px rgba(50, 205, 50, 0.7);
        }
        #stopBtn {
            background-color: #ff4500; /* OrangeRed */
            color: #0f0f1d;
        }
        #stopBtn:hover {
            background-color: #e03c00;
            box-shadow: 0 0 20px rgba(255, 69, 0, 0.7);
        }
        #allowOnceBtn {
            background-color: #ff8c00; /* Orange */
            color: #fff;
        }
        #allowOnceBtn:hover {
            background-color: #cc7000;
            box-shadow: 0 0 20px rgba(255, 140, 0, 0.7);
        }
        .loading-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10;
            border-radius: 12px;
            font-size: 1.5em;
            color: #ff8c00;
            display: none; /* Hidden by default */
        }
        .loading-overlay.visible {
            display: flex;
        }
        .loader {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #ff8c00;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>BLS Remote Control Panel - Bilal Work</h1>
        <div class="status-display">Current Status: <span id="currentStatus" class="status-text <?php echo $status == 1 ? 'active' : ''; ?>"><?php echo $status == 1 ? 'ALLOWED' : 'STOPPED'; ?></span></div>
        <div class="buttons-group">
            <button id="allowBtn">Allow</button>
            <button id="stopBtn">Stop</button>
            <button id="allowOnceBtn">Allow Once (5s)</button>
            <button id="delayedAllowBtn">Delayed Allow</button>
            <button id="changeCenterBtn" style="background-color: #32cd32;">Change Center</button>
        </div>
        <div style="margin-top:16px;">
            <div id="countdownContainer" style="display:none; align-items:center; gap:8px; justify-content:center;">
                <div id="countdownBar" style="width:200px; height:12px; background:#222; border-radius:6px; overflow:hidden;">
                    <div id="countdownFill" style="height:100%; width:0%; background:#32cd32;"></div>
                </div>
                <div id="countdownText" style="font-size:0.9em; color:#cfd8dc;">0s</div>
            </div>
        </div>
        <div style="margin-top:12px; padding:8px; background:#321a12; border-radius:6px; font-size:0.9em;">
            Current Center: <span id="currentCenter" style="color:#ff8c00; font-weight:bold; text-transform:capitalize;">Loading...</span>
        </div>
        <div style="position: absolute; top: 12px; right: 12px;">
            <button id="configureBtn" style="padding:6px 10px; font-size:0.9em;">Configure</button>
        </div>
        <div id="loadingOverlay" class="loading-overlay">
            <div class="loader"></div> Processing...
        </div>
    </div>

    <script>
        const currentStatusSpan = document.getElementById('currentStatus');
        const allowBtn = document.getElementById('allowBtn');
        const stopBtn = document.getElementById('stopBtn');
        const allowOnceBtn = document.getElementById('allowOnceBtn');
        const loadingOverlay = document.getElementById('loadingOverlay');
        const delayedAllowBtn = document.getElementById('delayedAllowBtn');
        const configureBtn = document.getElementById('configureBtn');

        // Create toast container
        (function createToastContainer() {
            const c = document.createElement('div');
            c.id = 'toastContainer';
            c.style.position = 'fixed';
            c.style.right = '12px';
            c.style.bottom = '12px';
            c.style.zIndex = '9999';
            document.body.appendChild(c);
        })();

        function showToast(message, type = 'info', timeout = 3000) {
            const c = document.getElementById('toastContainer');
            const t = document.createElement('div');
            t.textContent = message;
            t.style.background = type === 'error' ? '#ff4d4f' : '#2b2b2b';
            t.style.color = '#fff';
            t.style.padding = '10px 14px';
            t.style.marginTop = '8px';
            t.style.borderRadius = '6px';
            t.style.boxShadow = '0 6px 14px rgba(0,0,0,0.3)';
            t.style.fontSize = '0.95em';
            c.appendChild(t);
            setTimeout(() => { t.style.opacity = '0'; setTimeout(() => t.remove(), 400); }, timeout);
        }

        // Improved configure modal with toggle for delayed_enabled
        function openConfigureModal(settings) {
            const modal = document.createElement('div');
            modal.style.position = 'fixed';
            modal.style.left = '0';
            modal.style.top = '0';
            modal.style.width = '100%';
            modal.style.height = '100%';
            modal.style.display = 'flex';
            modal.style.alignItems = 'center';
            modal.style.justifyContent = 'center';
            modal.style.background = 'rgba(0,0,0,0.6)';
            modal.style.zIndex = '100050';
            modal.innerHTML = `
                <div style="background:#1d0f0a; padding:20px; border-radius:8px; width:360px; border:1px solid #764a21; position:relative; z-index:100051;">
                    <h3 style="margin-top:0; color:#ff8c00;">Configure Server Behaviour</h3>
                    <label style="display:block; margin-bottom:10px;">Enable Delayed Allow:<br>
                        <input id="cfgDelayedEnabled" type="checkbox" ${settings.delayed_enabled ? 'checked' : ''} style="margin-top:8px;" />
                    </label>
                    <label style="display:block; margin-bottom:8px;">Delayed Allow (ms):<br><input id="cfgDelayedMs" type="number" value="${settings.delayed_ms}" style="width:100%; padding:8px; margin-top:6px;"/></label>
                    <label style="display:block; margin-bottom:8px;">Allow Once Duration (ms):<br><input id="cfgAllowOnceMs" type="number" value="${settings.allow_once_duration_ms}" style="width:100%; padding:8px; margin-top:6px;"/></label>
                    <div style="display:flex; gap:8px; justify-content:flex-end; margin-top:12px;">
                        <button id="cfgSave" style="background:#32cd32; color:#0f0f1d; padding:8px 12px; border-radius:6px;">Save</button>
                        <button id="cfgCancel" style="background:#ff4500; color:#0f0f1d; padding:8px 12px; border-radius:6px;">Cancel</button>
                    </div>
                </div>
            `;
        document.body.appendChild(modal);

            document.getElementById('cfgCancel').addEventListener('click', () => modal.remove());
            document.getElementById('cfgSave').addEventListener('click', async () => {
                const delayedMs = parseInt(document.getElementById('cfgDelayedMs').value, 10) || 1300;
                const allowOnceMs = parseInt(document.getElementById('cfgAllowOnceMs').value, 10) || 5000;
                const delayedEnabled = document.getElementById('cfgDelayedEnabled').checked ? 1 : 0;
                // Send configure request
                showLoading();
                try {
                    // first update the delays
                    const res = await fetch('action.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: `action=configure&delayed_ms=${delayedMs}&allow_once_duration_ms=${allowOnceMs}`
                    });
                    const data = await res.json();
                    if (!data.success) {
                        showToast('Failed to save configuration: ' + data.message, 'error');
                        return;
                    }

                    // then set delayed_enabled on/off
                    const res2 = await fetch('action.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: delayedEnabled ? 'action=delayed_allow' : 'action=disable_delayed'
                    });
                    const data2 = await res2.json();
                    if (data2.success) {
                        showToast('Configuration saved', 'info');
                        modal.remove();
                    } else {
                        showToast('Failed to update delayed flag: ' + data2.message, 'error');
                    }
                } catch (err) {
                    showToast('Network error while saving configuration', 'error');
                } finally { hideLoading(); }
            });
        }

        // Change Center modal
        function openChangeCenterModal(currentCenter) {
            const modal = document.createElement('div');
            modal.style.position = 'fixed';
            modal.style.left = '0';
            modal.style.top = '0';
            modal.style.width = '100%';
            modal.style.height = '100%';
            modal.style.display = 'flex';
            modal.style.alignItems = 'center';
            modal.style.justifyContent = 'center';
            modal.style.background = 'rgba(0,0,0,0.6)';
            modal.style.zIndex = '100050';
            const centers = ['karachi', 'lahore', 'islamabad'];
            const centerButtons = centers.map(c => 
                `<button data-center="${c}" style="width:100%; padding:12px; margin:8px 0; background:${c === currentCenter ? '#cc7000' : '#4a2d1a'}; color:#fff; border:1px solid ${c === currentCenter ? '#ff8c00' : '#764a21'}; border-radius:6px; cursor:pointer; font-size:1.05em; text-transform:capitalize;">${c}</button>`
            ).join('');
            modal.innerHTML = `
                <div style="background:#1d0f0a; padding:20px; border-radius:8px; width:320px; border:1px solid #764a21; position:relative; z-index:100051;">
                    <h3 style="margin-top:0; color:#ff8c00;">Select Visa Center</h3>
                    <div id="centerButtons">${centerButtons}</div>
                    <div style="margin-top:12px;">
                        <button id="centerCancel" style="width:100%; background:#ff4500; color:#0f0f1d; padding:8px 12px; border-radius:6px; cursor:pointer;">Cancel</button>
                    </div>
                </div>
            `;
            document.body.appendChild(modal);

            document.getElementById('centerCancel').addEventListener('click', () => modal.remove());
            document.querySelectorAll('[data-center]').forEach(btn => {
                btn.addEventListener('click', async (e) => {
                    const center = e.target.dataset.center;
                    showLoading();
                    try {
                        const res = await fetch('action.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: `action=set_center&center=${center}`
                        });
                        const data = await res.json();
                        if (data.success) {
                            showToast(`Center changed to ${center.toUpperCase()}`, 'info');
                            document.getElementById('currentCenter').textContent = center;
                            modal.remove();
                        } else {
                            showToast('Failed to change center: ' + data.message, 'error');
                        }
                    } catch (err) {
                        showToast('Network error while changing center', 'error');
                    } finally { hideLoading(); }
                });
            });
        }

    let allowOnceTimer = null; // To store the timer ID for "Allow Once"
    let allowOnceCountdownInterval = null; // interval for updating UI countdown
    let allowOnceRemainingMs = 0;

        function showLoading() {
            loadingOverlay.classList.add('visible');
            allowBtn.disabled = true;
            stopBtn.disabled = true;
            allowOnceBtn.disabled = true;
        }

        function hideLoading() {
            loadingOverlay.classList.remove('visible');
            allowBtn.disabled = false;
            stopBtn.disabled = false;
            allowOnceBtn.disabled = false;
            delayedAllowBtn.disabled = false;
        }

        function updateStatusDisplay(status) {
            if (status === 1) {
                currentStatusSpan.textContent = 'ALLOWED';
                currentStatusSpan.classList.add('active');
            } else {
                currentStatusSpan.textContent = 'STOPPED';
                currentStatusSpan.classList.remove('active');
            }
        }

        async function sendAction(action, duration = 0) {
            showLoading();
            try {
                const response = await fetch('action.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=${action}`
                });
                const data = await response.json();
                if (data.success) {
                    // If any action succeeded, clear client-side delayed behaviors
                    clearAllowOnceCountdown();
                    updateStatusDisplay(data.newStatus);
                    // Show action-specific toasts
                    switch (action) {
                        case 'allow':
                            showToast('Server set to ALLOWED (delays disabled)', 'info');
                            break;
                        case 'stop':
                            showToast('Server set to STOPPED', 'info');
                            break;
                        case 'allow_once':
                            showToast('Server ALLOWED temporarily for ' + Math.round((data.allow_once_duration_ms || duration * 1000) / 1000) + 's', 'info');
                            break;
                        case 'delayed_allow':
                            showToast('Server set to ALLOWED with delayed responses (' + (data.delayed_ms || '1300') + ' ms)', 'info');
                            break;
                        case 'configure':
                            showToast('Configuration updated', 'info');
                            break;
                        case 'fetch':
                            showToast('Fetched current configuration', 'info');
                            break;
                        default:
                            showToast('Action completed', 'info');
                    }

                    if (action === 'allow_once' && data.newStatus === 1) {
                        // Clear any previous "Allow Once" timer
                        if (allowOnceTimer) {
                            clearTimeout(allowOnceTimer);
                        }

                        // Determine duration in ms from server response (prefer server value)
                        const durationMs = data.allow_once_duration_ms || (duration * 1000);
                        allowOnceRemainingMs = durationMs;

                        // Start visual countdown
                        startAllowOnceCountdown(durationMs);

                        // Set a new timer to send 'stop' after the specified duration
                        allowOnceTimer = setTimeout(async () => {
                            await sendAction('stop'); // Send stop action after duration
                            console.log(`Sent 'stop' action after ${durationMs} ms (Allow Once timer).`);
                            allowOnceTimer = null; // Reset the timer ID
                        }, durationMs);
                    } else if (action === 'stop' && allowOnceTimer) {
                        // If 'stop' is manually clicked while an 'allow_once' timer is active, clear it.
                        clearTimeout(allowOnceTimer);
                        allowOnceTimer = null;
                        clearAllowOnceCountdown();
                        console.log("Allow Once timer cleared due to manual 'Stop' action.");
                    }
                } else {
                    alert('Error: ' + data.message);
                }
            } catch (error) {
                console.error('Fetch error:', error);
                alert('Network error or server unreachable.');
            } finally {
                hideLoading();
            }
        }

        allowBtn.addEventListener('click', () => {
            if (allowOnceTimer) { // If manual Allow is clicked, clear any pending Allow Once stop
                clearTimeout(allowOnceTimer);
                allowOnceTimer = null;
                clearAllowOnceCountdown();
            }
            sendAction('allow');
        });

        stopBtn.addEventListener('click', () => {
            if (allowOnceTimer) { // If manual Stop is clicked, clear any pending Allow Once stop
                clearTimeout(allowOnceTimer);
                allowOnceTimer = null;
                clearAllowOnceCountdown();
            }
            sendAction('stop');
        });

        delayedAllowBtn.addEventListener('click', () => {
            // When enabling delayed allow, also clear any client-side allow_once countdown
            if (allowOnceTimer) {
                clearTimeout(allowOnceTimer);
                allowOnceTimer = null;
                clearAllowOnceCountdown();
            }
            sendAction('delayed_allow');
        });

        allowOnceBtn.addEventListener('click', () => sendAction('allow_once', Math.round(<?php
            // Put default allow_once_duration_ms from server into JS in seconds
            $settings = getGlobalSettings();
            echo max(1, (int)($settings['allow_once_duration_ms'] / 1000));
        ?>)));

        function startAllowOnceCountdown(totalMs) {
            const container = document.getElementById('countdownContainer');
            const fill = document.getElementById('countdownFill');
            const text = document.getElementById('countdownText');
            container.style.display = 'flex';
            const start = Date.now();
            clearAllowOnceCountdown();
            allowOnceCountdownInterval = setInterval(() => {
                const elapsed = Date.now() - start;
                const pct = Math.min(100, (elapsed / totalMs) * 100);
                fill.style.width = pct + '%';
                const remaining = Math.max(0, totalMs - elapsed);
                text.textContent = (Math.ceil(remaining / 1000)) + 's';
                if (elapsed >= totalMs) {
                    clearAllowOnceCountdown();
                }
            }, 200);
        }

        function clearAllowOnceCountdown() {
            const container = document.getElementById('countdownContainer');
            const fill = document.getElementById('countdownFill');
            const text = document.getElementById('countdownText');
            if (allowOnceCountdownInterval) {
                clearInterval(allowOnceCountdownInterval);
                allowOnceCountdownInterval = null;
            }
            container.style.display = 'none';
            fill.style.width = '0%';
            text.textContent = '';
        }

        configureBtn.addEventListener('click', async () => {
            // fetch current settings from action.php by requesting a status update via action=fetch (fallback to status.php)
            try {
                const res = await fetch('action.php', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: 'action=fetch' });
                const data = await res.json();
                if (data && data.delayed_ms) {
                    openConfigureModal(data);
                } else {
                    // fallback to status.php
                    const s = await fetch('status.php');
                    const sd = await s.json();
                    openConfigureModal({ delayed_ms: sd.delayed_ms || 1300, allow_once_duration_ms: sd.allow_once_duration_ms || 5000 });
                }
            } catch (e) {
                openConfigureModal({ delayed_ms: 1300, allow_once_duration_ms: 5000 });
            }
        });

        const changeCenterBtn = document.getElementById('changeCenterBtn');
        changeCenterBtn.addEventListener('click', async () => {
            try {
                const res = await fetch('action.php', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: 'action=fetch' });
                const data = await res.json();
                const currentCenter = (data && data.visa_center) ? data.visa_center : 'islamabad';
                openChangeCenterModal(currentCenter);
            } catch (e) {
                openChangeCenterModal('islamabad');
            }
        });

        // Load and display current center on page load
        (async function loadCurrentCenter() {
            try {
                const res = await fetch('status.php');
                const data = await res.json();
                const center = (data && data.visa_center) ? data.visa_center : 'islamabad';
                document.getElementById('currentCenter').textContent = center;
            } catch (e) {
                document.getElementById('currentCenter').textContent = 'Error loading';
            }
        })();
    </script>
</body>
</html>
